/**********************************************************************************************

  Copyright (C) 2010 Hilscher Gesellschaft für Systemautomation mbH.
  This program can be used by everyone according the "industrialNETworX Public License INPL".
  The license can be downloaded under <http://www.industrialNETworX.com>.

***********************************************************************************************

   Last Modification:
    @version $Id: RPCTask_Public.h 17387 2011-05-31 12:52:00Z andreasme $

   Description:
     DCE-RPC packets & commands.

   Changes:

     Date        Author        Description
     ----------------------------------------------------------------------------------
     2010-03-23  AM            Created
**************************************************************************************/
#ifndef __RPC_TASK_PUBLIC_H
#define __RPC_TASK_PUBLIC_H

#include "uuid.h"


/**
 * @defgroup dcerpc_task DCERPC Server and Client Task
 * @ingroup tasks
 *
 * @brief This task provides DCERPC server and client services.
 */

/**
 * @brief DCERPC task queue name
 * @addtogroup dcerpc_task
 **/
#define DCERPC_TASK_QUEUE_NAME      "QUE_RPC_TASK"

/*************************************************************************************/
/* packet commands */

enum DCERPC_CMD_Ttag
{
  /* client side commands */
  DCERPC_CMD_CL_EP_BIND_REQ =        0x00000500,
  DCERPC_CMD_CL_EP_BIND_CNF,     /*  0x00000501 */
  DCERPC_CMD_CL_EP_UNBIND_REQ,   /*  0x00000502 */
  DCERPC_CMD_CL_EP_UNBIND_CNF,   /*  0x00000503 */

  DCERPC_CMD_SEND_CALL_REQ,     /*  0x00000504 */
  DCERPC_CMD_SEND_CALL_CNF,     /*  0x00000505 */

  /* server side commands */
  DCERPC_CMD_CL_EP_REGISTER_REQ =       0x00000510,
  DCERPC_CMD_CL_EP_REGISTER_CNF,     /* 0x00000511 */
  DCERPC_CMD_CL_EP_DEREGISTER_REQ,   /* 0x00000512 */
  DCERPC_CMD_CL_EP_DEREGISTER_CNF,   /* 0x00000513 */

  DCERPC_CMD_RECV_CALL_IND,            /*  0x00000514 */
  DCERPC_CMD_RECV_CALL_RSP,            /*  0x00000515 */

  /* internal commands */
  RPC_CMD_CL_UDP_TIMEOUT_IND =          0x00000530,
  RPC_CMD_CL_UDP_TIMEOUT_RSP,        /* 0x00000531 */
  RPC_CMD_CL_UDP_TIME_RELEASE_IND,   /* 0x00000532 */
  RPC_CMD_CL_UDP_TIME_RELEASE_RSP,   /* 0x00000533 */

  RPC_CMD_CHANNEL_INIT_REQ,          /* 0x00000542 */
  RPC_CMD_CHANNEL_INIT_CNF,          /* 0x00000543 */
  RPC_CMD_SHUTDOWN_IND,              /* 0x00000544 */
  RPC_CMD_SHUTDOWN_RSP,              /* 0x00000545 */

  /* administrative commands */
  DCERPC_REGISTER_FATAL_ERROR_CALLBACK_REQ,  /* 0x00000546 */
  DCERPC_REGISTER_FATAL_ERROR_CALLBACK_CNF,  /* 0x00000547 */

  DCERPC_UNREGISTER_FATAL_ERROR_CALLBACK_REQ, /* 0x00000548 */
  DCERPC_UNREGISTER_FATAL_ERROR_CALLBACK_CNF, /* 0x00000549 */
};


/*************************************************************************************/
/** @name fatal error callback function type definition
 *
 * The following type is used for fatal error callback function.
 * @{
 */
typedef TLR_VOID(*RPC_FATAL_ERROR_CLB)(TLR_UINT32 ulErrorCode, TLR_VOID* pvUserParam);

/** @} */


typedef enum RPC_STATUS_Etag                  RPC_STATUS_E; /**< RPC status code */

/** RPC status code */
enum RPC_STATUS_Etag                          /* RPC status code */
{
  /* success */
  RPC_S_OK                          = 0x00000000, /**< no error */
  RPC_E_RESPONSE_TO_BIG,
};

/***************************************************************************************/
/** @name PDU Data Representation
 *
 * The definition represents the several flags to specify the data representation of
 * CLRPC PDUs, also known as <code>RPCDRep</code>.
 * @{
 */

#if (CPU_FRMT == 0)
#define DCERPC_DREP_BYTE_ORDER                     (0x00)    /**< byte order in MOTOROLA format */
#elif (CPU_FRMT == 1)
#define DCERPC_DREP_BYTE_ORDER                     (0x10)    /**< byte order in INTEL format */
#else
#error  "unsupported CPU format, use CPU_FRMT = 0 for MOTOROLA, CPU_FRMT = 1 for INTEL"
#endif

#define DCERPC_DREP_CHAR_ASCII                     (0x00)    /**< characters in ASCII format */
#define DCERPC_DREP_CHAR_EBCDIC                    (0x01)    /**< characters in EBCDIC format */

#define DCERPC_DREP_FLOAT_IEEE                     (0x00)    /**< floating point IEEE conform */

/** @} */

/*************************************************************************************/
/* packets */

/** @name Packet Data Definition
 *
 * The following defined structures represent the specific parameter
 * of the several CLRPC services.
 * @{
 */



typedef TLR_UINT  (*DCERPC_CL_SIZE_CLB_T)(TLR_UINT uiOpcode, TLR_UINT uiLenData, CONST TLR_UINT8* pbData, CONST TLR_UINT8* abDRep);

/** DCERPC register endpoint data
 *
 * The structure contains parameters needed for registration of an endpoint.
 */
typedef struct DCERPC_CL_EP_REGISTER_REQ_DATA_Ttag    /* CLRPC server register request data */
{
  /** maximum receive len - the maximum request data, the server can receive */
  TLR_UINT          uiMaxRequestLen;

  /** pointer to function callback which gives dcerpc a hint
   * for the size to allocate, its is called on the first fragment of a fragmented request or
   * on a unfragmented request.*/
  DCERPC_CL_SIZE_CLB_T  fnSizeClb;

  /** This member defines the port the server should listen on */
  TLR_UINT          uiPort;

  /** interface UUID to register */
  UUID_T            tUuidInt;

  /** dcerpc interface version */
  TLR_UINT32        ulVersion;

  /** object uuid of this endpoint */
  UUID_T            tUuidObj;

  /** Reserved */
  TLR_UINT          uiReserved;

  /** array containing the annotation of the server
   *  maximum allowed annotation length is 64. It should be
   *  a zero terminated string! */
  TLR_UINT8         abAnnotation[64];

} DCERPC_CL_EP_REGISTER_REQ_DATA_T;


/** DCERPC request Data
 *
 * The structure contains parameter to notify a CLRPC receive indication.
 */
typedef struct DCERPC_REQUEST_DATA_Ttag        /* CLRPC server receive indication data */
{
  /** operation code of request */
  TLR_UINT          uiOpcode;

  /** senders ip address, unused on client side */
  TLR_UINT32        ulSenderIpAddr;

  /** maximum allowed call body length*/
  TLR_UINT          uiMaxLen;

  /** ndr encoding */
  TLR_UINT8         abDRep[3];
} DCERPC_REQUEST_DATA_T;

/** DCERPC response Data
 *
 * The structure contains parameter to notify a CLRPC receive indication.
 */
typedef struct DCERPC_RESPONSE_DATA_Ttag        /* CLRPC server receive indication data */
{
  /** operation code of request */
  TLR_UINT          uiOpcode;

  /** status returned in faults/rejects on client side, unused on server's side*/
  TLR_UINT32        ulRPCStatus;

  /** time the corresponding activity should be keept
   *  in rpc memory */
  TLR_UINT          uiActivityIdleTimeout;

  /** ndr encoding */
  TLR_UINT8         abDRep[3];
} DCERPC_RESPONSE_DATA_T;



/** DCERPC connectionless endpoint bind data
 *
 * The structure contains parameter, which are required by CLRPC stack to
 * register a CLRPC client.
 */
typedef struct DCERPC_CL_EP_BIND_REQ_DATA_Ttag    /* CLRPC client register request data */
{
  /** IP address of remote peer on network. */
  TLR_UINT32        ulIpAddr;

  /** UDP port of remote peer on network*/
  TLR_UINT          uiPort;

  /** The interface UUID of objects interface */
  UUID_T            tUuidIntf;

  /** The interface version to use*/
  TLR_UINT32        ulInterfaceVersion;

  /** the object's UUID */
  UUID_T            tUuidObj;
} DCERPC_CL_EP_BIND_REQ_DATA_T;


/** @} */


/** @name Packet Definition
 *
 * The following definied structures represent the packets of the several
 * CLRPC services.
 * @{
 */


typedef struct DCERPC_CL_EP_BIND_REQ_Ttag       /* CLRPC  deregister request */
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_CL_EP_BIND_REQ_DATA_T        tData;
} DCERPC_CL_EP_BIND_REQ_T;

typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_BIND_CNF_T;

typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_UNBIND_REQ_T;
typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_UNBIND_CNF_T;

typedef struct DCERPC_SEND_CALL_REQ_Ttag
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_REQUEST_DATA_T               tData;
} DCERPC_SEND_CALL_REQ_T;

typedef struct DCERPC_SEND_CALL_CNF_Ttag
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_RESPONSE_DATA_T              tData;
} DCERPC_SEND_CALL_CNF_T;


typedef struct DCERPC_CL_EP_REGISTER_REQ_Ttag       /* CLRPC  deregister request */
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_CL_EP_REGISTER_REQ_DATA_T    tData;
} DCERPC_CL_EP_REGISTER_REQ_T;

typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_REGISTER_CNF_T;

typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_DEREGISTER_REQ_T;
typedef TLR_EMPTY_PACKET_T DCERPC_CL_EP_DEREGISTER_CNF_T;



typedef struct DCERPC_RECV_CALL_IND_Ttag
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_REQUEST_DATA_T               tData;
} DCERPC_RECV_CALL_IND_T;

typedef struct DCERPC_RECV_CALL_RSP_Ttag
{
  /** packet header */
  TLR_PACKET_HEADER_T                 tHead;
  /** packet request data */
  DCERPC_RESPONSE_DATA_T              tData;
} DCERPC_RECV_CALL_RSP_T;


/** @} */


/** @name Packet Union
 *
 * The following unions combines the request packets and the confirmation packets
 * of a CLRPC service and combines the indication packets and the response packets
 * of a CLRPC service.
 * @{
 */



typedef union
{
  /** packet request */
  DCERPC_CL_EP_BIND_REQ_T         tReq;
  /** packet confirmation */
  DCERPC_CL_EP_BIND_CNF_T         tCnf;
} DCERPC_CL_EP_BIND_PCK_T;

typedef union
{
  /** packet request */
  DCERPC_CL_EP_UNBIND_REQ_T         tReq;
  /** packet confirmation */
  DCERPC_CL_EP_UNBIND_CNF_T         tCnf;
} DCERPC_CL_EP_UNBIND_PCK_T;

typedef union
{
  /** packet request */
  DCERPC_SEND_CALL_REQ_T         tReq;
  /** packet confirmation */
  DCERPC_SEND_CALL_CNF_T         tCnf;
} DCERPC_SEND_CALL_PCK_T;

typedef union
{
  /** packet request */
  DCERPC_CL_EP_REGISTER_REQ_T         tReq;
  /** packet confirmation */
  DCERPC_CL_EP_REGISTER_CNF_T         tCnf;
} DCERPC_CL_EP_REGISTER_PCK_T;

typedef union
{
  /** packet request */
  DCERPC_CL_EP_DEREGISTER_REQ_T         tReq;
  /** packet confirmation */
  DCERPC_CL_EP_DEREGISTER_CNF_T         tCnf;
} DCERPC_CL_EP_DEREGISTER_PCK_T;

typedef union
{
  /** packet request */
  DCERPC_RECV_CALL_IND_T         tInd;
  /** packet confirmation */
  DCERPC_RECV_CALL_RSP_T         tRsp;
} DCERPC_RECV_CALL_PCK_T;

/*
 * REGISTER FATAL ERROR CALLBACK SERVICE
 */

/* Request packet */
typedef __PACKED_PRE struct DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_DATA_Ttag
{
  RPC_FATAL_ERROR_CLB  pfnClbFnc;
  TLR_VOID*            pvUserParam;
} __PACKED_POST DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_DATA_T;

typedef struct DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_Ttag
{
  TLR_PACKET_HEADER_T                       tHead;
  DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_DATA_T   tData;
} DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_T;

/* Confirmation packet */
typedef TLR_EMPTY_PACKET_T                  DCERPC_REG_FATAL_ERROR_CALLBACK_CNF_T;

/* packet union */
typedef union DCERPC_REG_FATAL_ERROR_CALLBACK_PCK_Ttag
{
  DCERPC_REG_FATAL_ERROR_CALLBACK_REQ_T        tReq;
  DCERPC_REG_FATAL_ERROR_CALLBACK_CNF_T        tCnf;
} DCERPC_REG_FATAL_ERROR_CALLBACK_PCK_T;


/*
 * UNREGISTER FATAL ERROR CALLBACK SERVICE
 */

/* Request packet */
typedef TLR_EMPTY_PACKET_T                  DCERPC_UNREG_FATAL_ERROR_CALLBACK_REQ_T;

/* Confirmation packet */
typedef TLR_EMPTY_PACKET_T                  DCERPC_UNREG_FATAL_ERROR_CALLBACK_CNF_T;

/* packet union */
typedef union DCERPC_UNREG_FATAL_ERROR_CALLBACK_PCK_Ttag
{
  DCERPC_UNREG_FATAL_ERROR_CALLBACK_REQ_T      tReq;
  DCERPC_UNREG_FATAL_ERROR_CALLBACK_CNF_T      tCnf;
} DCERPC_UNREG_FATAL_ERROR_CALLBACK_PCK_T;


/** @} */

/* this next section is taken from file PNIO_spec.h */
/* it is directly connected to RPC and therefore the */
/* definitions shall be located in RPC - not in Profinet */





/*************************************************************************************/
#endif /* #ifndef __RPC_TASK_PUBLIC_H */
